<?php
/**
 * Segment - Verwaltung einzelner Backup-Segmente
 * 
 * Ein Segment ist ein Teil des Backups (z.B. uploads-001.zip).
 * Segmente werden direkt beim Backup gefüllt (Streaming),
 * sodass kein nachträgliches Zusammenfügen nötig ist.
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Core;

if (!defined('ABSPATH')) {
    exit;
}

class Segment {
    
    /** @var int Maximale Segment-Größe (100 MB) */
    const MAX_SIZE = 104857600;
    
    /** @var string Segment-Name */
    private $name;
    
    /** @var string Segment-Typ */
    private $type;
    
    /** @var string Pfad zur Segment-Datei */
    private $path;
    
    /** @var \ZipArchive ZIP-Handle */
    private $zip;
    
    /** @var int Aktuelle Größe */
    private $current_size = 0;
    
    /** @var int Anzahl Dateien */
    private $file_count = 0;
    
    /** @var bool Ob Segment offen ist */
    private $is_open = false;
    
    /** @var array Dateien in diesem Segment */
    private $files = [];
    
    /**
     * Erstellt ein neues Segment
     * 
     * @param string $name Segment-Name (z.B. 'uploads-001')
     * @param string $type Typ ('database', 'uploads', 'themes', 'plugins')
     * @param string $base_path Basis-Verzeichnis für Segmente
     */
    public function __construct(string $name, string $type, string $base_path) {
        $this->name = $name;
        $this->type = $type;
        $this->path = trailingslashit($base_path) . $name . '.zip';
    }
    
    /**
     * Öffnet das Segment zum Schreiben
     * 
     * @return bool
     * @throws \Exception
     */
    public function open(): bool {
        if ($this->is_open) {
            return true;
        }
        
        $this->zip = new \ZipArchive();
        $result = $this->zip->open($this->path, \ZipArchive::CREATE | \ZipArchive::OVERWRITE);
        
        if ($result !== true) {
            throw new \Exception("Konnte Segment nicht erstellen: {$this->path} (Error: $result)");
        }
        
        // Kompression auf Maximum setzen
        $this->zip->setCompressionIndex(0, \ZipArchive::CM_DEFLATE);
        
        $this->is_open = true;
        return true;
    }
    
    /**
     * Fügt eine Datei zum Segment hinzu
     * 
     * @param string $source_path Quellpfad
     * @param string $archive_path Pfad im Archiv
     * @return bool
     */
    public function addFile(string $source_path, string $archive_path): bool {
        if (!$this->is_open) {
            $this->open();
        }
        
        if (!file_exists($source_path)) {
            return false;
        }
        
        $size = filesize($source_path);
        
        // Prüfen ob Segment voll ist
        if ($this->current_size + $size > self::MAX_SIZE && $this->file_count > 0) {
            return false; // Signal: Neues Segment starten
        }
        
        $result = $this->zip->addFile($source_path, $archive_path);
        
        if ($result) {
            $this->current_size += $size;
            $this->file_count++;
            $this->files[$archive_path] = [
                'size' => $size,
                'checksum' => hash_file('sha256', $source_path),
            ];
        }
        
        return $result;
    }
    
    /**
     * Fügt String-Inhalt als Datei hinzu
     * 
     * @param string $content Inhalt
     * @param string $archive_path Pfad im Archiv
     * @return bool
     */
    public function addFromString(string $content, string $archive_path): bool {
        if (!$this->is_open) {
            $this->open();
        }
        
        $size = strlen($content);
        
        $result = $this->zip->addFromString($archive_path, $content);
        
        if ($result) {
            $this->current_size += $size;
            $this->file_count++;
            $this->files[$archive_path] = [
                'size' => $size,
                'checksum' => hash('sha256', $content),
            ];
        }
        
        return $result;
    }
    
    /**
     * Schließt das Segment
     * 
     * @return array Segment-Info für Manifest
     */
    public function close(): array {
        if ($this->is_open && $this->zip) {
            $this->zip->close();
            $this->is_open = false;
        }
        
        $final_size = file_exists($this->path) ? filesize($this->path) : 0;
        $checksum = file_exists($this->path) ? hash_file('sha256', $this->path) : '';
        
        return [
            'name' => $this->name,
            'type' => $this->type,
            'path' => $this->path,
            'size' => $final_size,
            'uncompressed_size' => $this->current_size,
            'file_count' => $this->file_count,
            'checksum' => $checksum,
            'files' => $this->files,
        ];
    }
    
    /**
     * Prüft ob Segment voll ist
     * 
     * @return bool
     */
    public function isFull(): bool {
        return $this->current_size >= self::MAX_SIZE;
    }
    
    /**
     * Gibt den aktuellen Füllstand zurück
     * 
     * @return float Prozent (0-100)
     */
    public function getFillLevel(): float {
        return ($this->current_size / self::MAX_SIZE) * 100;
    }
    
    /**
     * Getter für Name
     */
    public function getName(): string {
        return $this->name;
    }
    
    /**
     * Getter für Typ
     */
    public function getType(): string {
        return $this->type;
    }
    
    /**
     * Getter für Pfad
     */
    public function getPath(): string {
        return $this->path;
    }
    
    /**
     * Getter für aktuelle Größe
     */
    public function getCurrentSize(): int {
        return $this->current_size;
    }
    
    /**
     * Getter für Datei-Anzahl
     */
    public function getFileCount(): int {
        return $this->file_count;
    }
    
    // ========================================
    // RESTORE-METHODEN
    // ========================================
    
    /**
     * Öffnet ein bestehendes Segment zum Lesen
     * 
     * @return bool
     */
    public function openForReading(): bool {
        if (!file_exists($this->path)) {
            throw new \Exception("Segment nicht gefunden: {$this->path}");
        }
        
        $this->zip = new \ZipArchive();
        $result = $this->zip->open($this->path, \ZipArchive::RDONLY);
        
        if ($result !== true) {
            throw new \Exception("Konnte Segment nicht öffnen: {$this->path}");
        }
        
        $this->is_open = true;
        $this->file_count = $this->zip->numFiles;
        
        return true;
    }
    
    /**
     * Extrahiert eine einzelne Datei
     * 
     * @param string $archive_path Pfad im Archiv
     * @param string $destination Zielpfad
     * @return bool
     */
    public function extractFile(string $archive_path, string $destination): bool {
        if (!$this->is_open) {
            $this->openForReading();
        }
        
        // Verzeichnis erstellen falls nötig
        $dir = dirname($destination);
        if (!is_dir($dir)) {
            wp_mkdir_p($dir);
        }
        
        // Datei extrahieren
        $content = $this->zip->getFromName($archive_path);
        if ($content === false) {
            return false;
        }
        
        return file_put_contents($destination, $content) !== false;
    }
    
    /**
     * Extrahiert alle Dateien
     * 
     * @param string $destination Zielverzeichnis
     * @param callable|null $progress_callback Callback für Fortschritt
     * @return int Anzahl extrahierter Dateien
     */
    public function extractAll(string $destination, ?callable $progress_callback = null): int {
        if (!$this->is_open) {
            $this->openForReading();
        }
        
        $extracted = 0;
        
        for ($i = 0; $i < $this->zip->numFiles; $i++) {
            $stat = $this->zip->statIndex($i);
            $filename = $stat['name'];
            
            // Verzeichnisse überspringen
            if (substr($filename, -1) === '/') {
                continue;
            }
            
            $target = trailingslashit($destination) . $filename;
            
            // Verzeichnis erstellen
            $dir = dirname($target);
            if (!is_dir($dir)) {
                wp_mkdir_p($dir);
            }
            
            // Datei extrahieren
            $content = $this->zip->getFromIndex($i);
            if ($content !== false) {
                file_put_contents($target, $content);
                $extracted++;
            }
            
            // Fortschritt melden
            if ($progress_callback) {
                $progress_callback($extracted, $this->file_count, $filename);
            }
        }
        
        return $extracted;
    }
    
    /**
     * Gibt den Inhalt einer Datei zurück ohne zu extrahieren
     * 
     * @param string $archive_path Pfad im Archiv
     * @return string|false
     */
    public function getContent(string $archive_path) {
        if (!$this->is_open) {
            $this->openForReading();
        }
        
        return $this->zip->getFromName($archive_path);
    }
    
    /**
     * Listet alle Dateien im Segment
     * 
     * @return array
     */
    public function listFiles(): array {
        if (!$this->is_open) {
            $this->openForReading();
        }
        
        $files = [];
        for ($i = 0; $i < $this->zip->numFiles; $i++) {
            $stat = $this->zip->statIndex($i);
            $files[] = [
                'name' => $stat['name'],
                'size' => $stat['size'],
                'compressed_size' => $stat['comp_size'],
            ];
        }
        
        return $files;
    }
    
    /**
     * Verifiziert Segment-Integrität
     * 
     * @param string $expected_checksum Erwartete Checksumme
     * @return bool
     */
    public function verify(string $expected_checksum): bool {
        if (!file_exists($this->path)) {
            return false;
        }
        
        $actual = hash_file('sha256', $this->path);
        return hash_equals($expected_checksum, $actual);
    }
}

